//
// (c) 2025 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0

import wesual.Ui.Private 1.0

FocusScope {
    id : uiSpinBox

    property alias value    : p_.value
    property alias decimals : p_.decimals
    property alias minimum  : p_.minimum
    property alias maximum  : p_.maximum
    property bool indeterminate : false
    property bool readOnly : false

    readonly property alias hovered : edit.hovered

    signal editingFinished()
    signal updateRequested(real newValue)

    function stepUp() {
        if (indeterminate)
            return;

        const newVal = p_.stepUp(p_.parseNumber(edit.editText()));
        if (newVal !== Number.NaN) {
            uiSpinBox.updateRequested(newVal);
        }
    }

    function stepDown() {
        if (indeterminate)
            return;

        const newVal = p_.stepDown(p_.parseNumber(edit.editText()));
        if (newVal !== Number.NaN) {
            uiSpinBox.updateRequested(newVal);
        }
    }

    implicitWidth : edit.implicitWidth
    implicitHeight : edit.implicitHeight

    Keys.onUpPressed   : stepUp()
    Keys.onDownPressed : stepDown()

    UiSpinBoxExPrivate {
        id : p_
    }

    UiLineEdit {
        id : edit

        function commit(text) {
            if (acceptableInput) {
                var value = p_.parseNumber(text);
                if (value !== Number.NaN) {
                    uiSpinBox.updateRequested(value);
                }
            }
        }

        validator : DoubleValidator {
            bottom : p_.minimum
            top : p_.maximum
            decimals : p_.decimals
            locale : Qt.locale().name
            notation : DoubleValidator.StandardNotation
        }

        anchors.fill : uiSpinBox
        bindable : true
        focus : true
        text : !indeterminate ? p_.displayString : ""
        minimumWidth : 64
        rightPadding : 30
        readOnly : uiSpinBox.readOnly

        onEditingFinished : uiSpinBox.editingFinished()
        onUpdateRequested : commit(newValue)
    }

    Timer {
        id : stepTimer_

        readonly property int  minTick : 15
        readonly property int  maxTick : 250

        property int nextTick : 0
        property int prevTick : maxTick

        property MouseArea trigger
        property real currentValue

        running : trigger && trigger.canStep && trigger.containsPress
        interval : 10
        repeat : true

        triggeredOnStart : true

        onRunningChanged : {
            nextTick = 0;
            prevTick = maxTick;
        }

        onTriggered : {
            nextTick -= interval;

            if (nextTick >= 0)
                return;

            currentValue = trigger.step(currentValue);

            uiSpinBox.updateRequested(currentValue);

            prevTick = Math.max(minTick, prevTick * .8);
            nextTick = prevTick;
        }
    }

    MouseArea {
        id : up

        readonly property bool canStep :
            enabled && !indeterminate && p_.value < p_.maximum

        function step(value) {
            return p_.stepUp(value);
        }

        width : 20
        height : 11
        enabled : !uiSpinBox.readOnly
        cursorShape : canStep ? Qt.PointingHandCursor : Qt.ArrowCursor
        hoverEnabled : true

        anchors {
            right : uiSpinBox.right
            top : uiSpinBox.top
            margins : 1
        }

        onPressed : {
            stepTimer_.currentValue = p_.parseNumber(edit.editText());
            stepTimer_.trigger = up;
        }

        Rectangle {
            anchors.fill : up
            color : {
                if (!up.canStep) {
                    return UiColors.getColor(UiColors.White);
                } else if (up.containsPress) {
                    return UiColors.getColor(UiColors.SelectionGreen);
                } else if (up.containsMouse) {
                    return UiColors.getColor(UiColors.HoverGreen);
                } else {
                    return UiColors.getColor(UiColors.White);
                }
            }
        }

        UiColorizedImage {
            id : arrowUp

            source : "qrc:/ui/images/spinbox-arrow-up.png"
            color  : {
                if (!up.canStep) {
                    return UiColors.getColor(UiColors.DisabledLightGrey13);
                } else if (up.containsPress) {
                    return UiColors.getColor(UiColors.White);
                } else {
                    return "transparent";
                }
            }
            x : 6
            anchors {
                bottom : up.bottom
                bottomMargin : 3
            }
        }
    }

    MouseArea {
        id : down

        readonly property bool canStep :
            enabled && !indeterminate && p_.value > p_.minimum

        function step(value) {
            return p_.stepDown(value);
        }

        width : 20
        height : 11
        enabled : !uiSpinBox.readOnly
        cursorShape : canStep ? Qt.PointingHandCursor : Qt.ArrowCursor
        hoverEnabled : true

        anchors {
            right : uiSpinBox.right
            bottom : uiSpinBox.bottom
            margins : 1
        }

        onPressed : {
            stepTimer_.currentValue = p_.parseNumber(edit.editText());
            stepTimer_.trigger = down;
        }

        Rectangle {
            anchors.fill : down
            color : {
                if (!down.canStep) {
                    return UiColors.getColor(UiColors.White);
                } else if (down.containsPress) {
                    return UiColors.getColor(UiColors.SelectionGreen);
                } else if (down.containsMouse) {
                    return UiColors.getColor(UiColors.HoverGreen);
                } else {
                    return UiColors.getColor(UiColors.White);
                }
            }
        }

        UiColorizedImage {
            id : arrowDown

            source : "qrc:/ui/images/spinbox-arrow-down.png"
            color  : {
                if (!down.canStep) {
                    return UiColors.getColor(UiColors.DisabledLightGrey13);
                } else if (down.containsPress) {
                    return UiColors.getColor(UiColors.White);
                } else {
                    return "transparent";
                }
            }
            x : 6
            anchors {
                top : down.top
                topMargin : 3
            }
        }
    }
}
